"use client";
import { EyeCloseIcon, EyeIcon } from "@/assets/images";
import Image from "next/image";
import Link from "next/link";
import React, { useState } from "react";
import axios from "axios";

export default function SignUpForm() {
  const [showPassword, setShowPassword] = useState(false);
  const [isChecked, setIsChecked] = useState(false);
  const [form, setForm] = useState({
    name: "",
    email: "",
    password: "",
    confirmPassword: "",
    userType: "",
  });
  const [submitting, setSubmitting] = useState(false);
  const [message, setMessage] = useState("");

  const apiUrl = process.env.NEXT_PUBLIC_API_URL;

  const handleChange = (e) => {
    const { name, value } = e.target;
    setForm({ ...form, [name]: value });

    // Alphanumeric password check on change
    if (name === "password") {
      const trimmed = value.trim();
      if (trimmed && !/^[a-zA-Z0-9]+$/.test(trimmed)) {
        setMessage("Password must be alphanumeric only. Example: Auth1234");
      } else {
        setMessage("");
      }
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setMessage("");

    // Required fields check
    if (
      !form.name ||
      !form.email ||
      !form.password ||
      !form.confirmPassword ||
      !form.userType
    ) {
      setMessage("All fields are required.");
      return;
    }

    // Password match check
    if (form.password !== form.confirmPassword) {
      setMessage("Passwords do not match.");
      return;
    }

    // Terms & Conditions check
    if (!isChecked) {
      setMessage("You must agree to the Terms & Conditions and Privacy Policy.");
      return;
    }

    setSubmitting(true);
    try {
      const payload = {
        name: form.name,
        email: form.email,
        password: form.password.trim(),
        userType: form.userType,
      };
      const response = await axios.post(`${apiUrl}/auth/signup`, payload);
      setMessage(response.data.message || "Signup successful!");
    } catch (err) {
      setMessage(err.response?.data?.error || "Signup failed.");
    }
    setSubmitting(false);
  };

  return (
    <div className="flex flex-col flex-1 w-full md:w-1/2 py-10 md:py-12 md:h-screen md:overflow-y-auto scroll_bar">
      <div className="flex flex-col justify-center flex-1 max-w-md xl:max-w-lg w-full mx-auto px-5 md:px-8">
        <div>
          <div className="mb-7 sm:mb-10">
            <h3 className="mb-2 md:mb-3 font-semibold text-gray text-3xl lg:text-4xl xl:text-[42px] lg:leading-12">Create an Account</h3>
            <p className="text-sm lg:text-base xl:text-lg font-normal text-[#575757]">Please tell us bit about yourself so that we can personalize the experience. </p>
          </div>
          <div>
            <form className="max-w-md mx-auto w-full" onSubmit={handleSubmit}>
              <div className="space-y-6">
                <div>
                  <label className="mb-1.5 block text-sm font-medium text-gray-700 ">
                    Your Name
                  </label>
                  <div className="relative">
                    <input
                      placeholder="Enter your full name"
                      type="text"
                      name="name"
                      value={form.name}
                      onChange={handleChange}
                      className="h-[52px] w-full rounded-sm border appearance-none px-4 py-2.5 text-sm shadow-theme-xs placeholder:text-gray-400 focus:outline-hidden bg-transparent text-gray-800 border-gray-300 focus:border-brand-300 focus:ring-brand-500/20"
                      required
                    />
                  </div>
                </div>
                <div>
                  <label className="mb-1.5 block text-sm font-medium text-gray-700 ">
                    Email Address
                  </label>
                  <div className="relative">
                    <input
                      placeholder="Enter your email address"
                      type="email"
                      name="email"
                      value={form.email}
                      onChange={handleChange}
                      className="h-[52px] w-full rounded-sm border appearance-none px-4 py-2.5 text-sm shadow-theme-xs placeholder:text-gray-400 focus:outline-hidden bg-transparent text-gray-800 border-gray-300 focus:border-brand-300 focus:ring-brand-500/20"
                      required
                    />
                  </div>
                </div>
                <div>
                  <label className="mb-1.5 block text-sm font-medium text-gray-700 ">
                    User Type
                  </label>
                  <div className="relative">
                    <select
                      name="userType"
                      value={form.userType}
                      onChange={handleChange}
                      className="h-[52px] w-full rounded-sm border appearance-none px-4 py-2.5 text-sm shadow-theme-xs placeholder:text-gray-400 focus:outline-hidden bg-transparent text-gray-800 border-gray-300 focus:border-brand-300 focus:ring-brand-500/20"
                      required
                    >
                      <option value="">Select</option>
                      <option value="Au">Author</option>
                      <option value="Pu">Publisher</option>
                    </select>
                  </div>
                </div>
                <div>
                  <label className="mb-1.5 block text-sm font-medium text-gray-700 ">
                    Password
                  </label>
                  <div className="relative">
                    <input
                      type={showPassword ? "text" : "password"}
                      placeholder="Enter your password"
                      name="password"
                      value={form.password}
                      onChange={handleChange}
                      className="h-[52px] w-full rounded-sm border appearance-none pl-4 pr-12 py-2.5 text-sm shadow-theme-xs placeholder:text-gray-400 focus:outline-hidden bg-transparent text-gray-800 border-gray-300 focus:border-brand-300 focus:ring-brand-500/20"
                      required
                    />
                    <span onClick={() => setShowPassword(!showPassword)} className="absolute z-30 opacity-50 -translate-y-1/2 cursor-pointer right-4 top-1/2">
                      {showPassword ? (
                        <Image width={24} height={24} src={EyeIcon} alt="Show" />
                      ) : (
                        <Image width={24} height={24} src={EyeCloseIcon} alt="Hide" />
                      )}
                    </span>
                  </div>
                </div>
                <div>
                  <label className="mb-1.5 block text-sm font-medium text-gray-700 ">
                    Confirm Password
                  </label>
                  <div className="relative">
                    <input
                      type={showPassword ? "text" : "password"}
                      placeholder="Re-enter Your Password"
                      name="confirmPassword"
                      value={form.confirmPassword}
                      onChange={handleChange}
                      className="h-[52px] w-full rounded-sm border appearance-none pl-4 pr-12 py-2.5 text-sm shadow-theme-xs placeholder:text-gray-400 focus:outline-hidden bg-transparent text-gray-800 border-gray-300 focus:border-brand-300 focus:ring-brand-500/20"
                      required
                    />
                    <span onClick={() => setShowPassword(!showPassword)} className="absolute z-30 opacity-50 -translate-y-1/2 cursor-pointer right-4 top-1/2">
                      {showPassword ? (
                        <Image width={24} height={24} src={EyeIcon} alt="Show" />
                      ) : (
                        <Image width={24} height={24} src={EyeCloseIcon} alt="Hide" />
                      )}
                    </span>
                  </div>
                </div>
                <div className="flex items-center justify-start">
                  <div className="flex items-center gap-3">
                    <input
                      type="checkbox"
                      checked={isChecked}
                      onChange={() => setIsChecked(!isChecked)}
                      className="w-5 h-5 cursor-pointer border border-gray-300 checked:border-transparent rounded-sm checked:bg-brand-500 disabled:opacity-60"
                      required
                    />
                    <span className="block font-normal text-gray-700 text-sm ">
                      I agreed to the <Link href="/" className="text-primary">Terms & Conditions</Link> and <Link href="/" className="text-primary">Privacy Policy</Link>
                    </span>
                  </div>
                </div>
                <div className="flex flex-col items-center text-center pt-3">
                  <button
                    type="submit"
                    className="h-15 items-center justify-center text-base gap-2 rounded-sm w-full transition px-4 py-2.5 bg-primary text-white hover:bg-primaryDark cursor-pointer"
                    disabled={submitting}
                  >
                    {submitting ? "Creating Account..." : "Create Account"}
                  </button>
                  <p className="text-[13px] leading-5 text-gray py-3">Already have an account? <Link href="/login" className="text-primary">Sign In</Link></p>
                  {message && (
                    <div className="mt-2 text-center text-sm text-red-600">{message}</div>
                  )}
                </div>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}
