'use client'
import { useEffect, useState } from 'react';
import Layout from "@/components/Layout/LayoutAdmin";
import Image from "next/image";
import { search, colorview } from '@/assets/images';
import axios from "axios";
import { useAuth } from "@/context/AuthContext";
import { formatDateWithOrdinal } from "@/utils/dateFormat";

export default function ContentManagement() {
  const [results, setResults] = useState([]);
  const [currentPage, setCurrentPage] = useState(1);
  const resultsPerPage = 3;
  const [searchTerm, setSearchTerm] = useState('');
  const { user, loading } = useAuth();
  // console.log("user in ContentManagement:", user);
  
  useEffect(() => {
    const fetchData = async () => {
      const apiUrl = process.env.NEXT_PUBLIC_API_URL;
      try {
        const response = await axios.get(`${apiUrl}/getdata/admin`, {
          headers: {
            Authorization: `Bearer ${user?.token}`
          }
        });
        console.log("Fetched data:", response.data);
        setResults(response.data || []);
      } catch (err) {
        setResults([]);
      }
    };
    fetchData();
  }, [user]);

  // Filter results by title or user name
  const filteredResults = results.filter(result => {
    const title = result.title?.toLowerCase() || '';
    const author = result.author?.toLowerCase() || '';
    const userName = result.user?.name?.toLowerCase() || '';
    const search = searchTerm.toLowerCase();
    return (
      title.includes(search) ||
      author.includes(search) ||
      userName.includes(search)
    );
  });

  // Pagination logic
  const indexOfLastResult = currentPage * resultsPerPage;
  const indexOfFirstResult = indexOfLastResult - resultsPerPage;
  const currentResults = filteredResults.slice(indexOfFirstResult, indexOfLastResult);
  const totalPages = Math.ceil(filteredResults.length / resultsPerPage);

  const renderUserType = (user) => {
    if (!user) return '';
    const type = typeof user === 'object' ? user.userType : user;
    if (!type) return '';
    switch (type.toLowerCase()) {
      case 'publisher':
      case 'pu':
        return 'Publisher';
      case 'author':
      case 'au':
        return 'Author';
      default:
        return type;
    }
  };

  const handlePageChange = (page) => {
    if (page >= 1 && page <= totalPages) setCurrentPage(page);
  };

  const handleSearch = (e) => {
    setSearchTerm(e.target.value);
    setCurrentPage(1); // Reset to first page on search
  };

  return (
    <Layout>
      <div className="max-w-full mx-auto my-10">
        <h3 className="mb-5 font-semibold text-[#373737] text-2xl">Content Library</h3>
        <div className="border-b border-gray-200 mb-10">
          <div className="inline-flex relative">
            <button className="text-emerald-500 font-normal px-4 py-2">
              All Books
            </button>
            <span className="absolute bottom-0 left-0 h-1 w-full bg-emerald-500 rounded"></span>
          </div>
        </div>
        <div className="flex items-center w-full max-w-full space-x-2">
          <div className="flex items-center flex-grow border border-gray-300 rounded-md px-3">
            <Image src={search} width={'auto'} alt="search" />
            <input
              type="text"
              placeholder="Search by title or user name"
              className="w-full pl-2 text-sm placeholder-gray-400 focus:outline-none leading-[56px]"
              value={searchTerm}
              onChange={handleSearch}
            />
          </div>
        </div>

        <div className="table_wpr pb-5 mt-5">
          <div className="relative overflow-x-auto">
            <table className="glob_table">
              <thead>
                <tr>
                  <th>Sl. No</th>
                  <th>User Name</th>
                  <th>Content Title</th>
                  <th>No. of Scans</th>
                  <th className="text-center">Upload Date</th>
                  <th>User Type</th>
                  <th>Scan Result</th>
                </tr>
              </thead>
              <tbody>
                {currentResults.length === 0 ? (
                  <tr>
                    <td colSpan={7} className="text-center">No results found.</td>
                  </tr>
                ) : (
                  currentResults.map((result, r) => (
                    <tr key={r}>
                      <td className="max-w-32">{indexOfFirstResult + r + 1}</td>
                      <td className="max-w-32">{result.user?.name || ''}</td>
                      <td>{result.title}</td>
                      <td>{result.rescan}</td>
                      <td>{formatDateWithOrdinal(result.uploadDate)}</td>
                      <td>{renderUserType(result.user)}</td>
                      <td>
                        <a href={`/content-library-view?bookid=${result.itemid}`}>
                          <span className='flex items-center text-[#04B47D]'>
                            <Image src={colorview} className='me-2' alt="" />
                            View
                          </span>
                        </a>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>

        {/* Pagination Controls */}
        <div className="flex justify-end mt-8">
          <ul className="flex items-center space-x-2">
            <li
              className={`px-3 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${currentPage === 1 ? 'opacity-50 cursor-not-allowed' : ''}`}
              onClick={() => handlePageChange(currentPage - 1)}
            >
              Prev
            </li>
            {[...Array(totalPages)].map((_, idx) => (
              <li
                key={idx}
                className={`w-8 h-8 border border-[#c9c9c9] flex items-center justify-center ${currentPage === idx + 1 ? 'bg-primary text-white' : 'bg-white text-black'} text-sm font-medium rounded cursor-pointer`}
                onClick={() => handlePageChange(idx + 1)}
              >
                {idx + 1}
              </li>
            ))}
            <li
              className={`px-3 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${currentPage === totalPages ? 'opacity-50 cursor-not-allowed' : ''}`}
              onClick={() => handlePageChange(currentPage + 1)}
            >
              Next
            </li>
          </ul>
        </div>
      </div>
    </Layout>
  );
}