'use client'
import { useState, useEffect, useRef } from 'react';
import Layout from "@/components/Layout/LayoutAdmin";
import Image from "next/image";
import Link from "next/link";
import { droparrowdown, uprow, search, falseright } from '@/assets/images';
import { useAuth } from "@/context/AuthContext";
import { useSearchParams } from 'next/navigation';
import axios from 'axios';


export default function ContentLibraryView() {
  const searchParams = useSearchParams();
  const bookid = searchParams.get('bookid');
  const [open, setOpen] = useState(false);
  const [isOpen, setIsOpen] = useState(false);
  const modalRef = useRef(null);
  const [selectedStatus, setSelectedStatus] = useState('Review Status');
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const resultsPerPage = 10;
  const [results, setResults] = useState([]);
  const [loadingA, setLoadingA] = useState(false);
  const [book, setBook] = useState(null);

  const { user, loading } = useAuth();
  const authID = user ? user.userId : 0;
  const apiUrl = process.env.NEXT_PUBLIC_API_URL;

  // Fetch book details
  const fetchBook = async () => {
    try {
      const response = await axios.get(`${apiUrl}/getdata/adminbook?tab=uploads&authID=${authID}&keyName=_id&keyValue=${bookid}`, {
          headers: {
            Authorization: `Bearer ${user?.token}`
          }
        });
      console.log("Fetched book data:", response.data);
      
      setBook(response.data[0] || null);
    } catch (err) {
      setBook(null);
    }
  };

  // Fetch search results
  const fetchBookSearch = async () => {
    try {
      setLoadingA(true);
      const response = await axios.get(`${apiUrl}/getdata/searchresult?itemid=${bookid}`, {
          headers: {
            Authorization: `Bearer ${user?.token}`
          }
        });
      console.log("Fetched search results:", response.data);
      setResults(response.data || []);
    } catch (err) {
      setResults([]);
    }
    setLoadingA(false);
  };

  useEffect(() => {
    function handleClickOutside(event) {
      if (modalRef.current && !modalRef.current.contains(event.target)) {
        setIsOpen(false);
      }
    }
    if (!loading && user) {
      fetchBook();
      fetchBookSearch();
    }
    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
      document.body.classList.add("overflow-hidden");
    } else {
      document.body.classList.remove("overflow-hidden");
    }
    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
    // eslint-disable-next-line
  }, [isOpen, bookid, authID, loading, user]);

  // Filter results by keyword, video, and channel
  const filteredResults = results.filter(result => {
    const search = searchTerm.toLowerCase();
    return (
      result.keyword?.toLowerCase().includes(search) ||
      result.video?.toLowerCase().includes(search) ||
      result.channel?.name?.toLowerCase().includes(search)
    );
  });

  // Pagination logic (apply to filteredResults)
  const indexOfLastResult = currentPage * resultsPerPage;
  const indexOfFirstResult = indexOfLastResult - resultsPerPage;
  const currentResults = filteredResults.slice(indexOfFirstResult, indexOfLastResult);
  const totalPages = Math.ceil(filteredResults.length / resultsPerPage);

  const handlePageChange = (page) => {
    if (page >= 1 && page <= totalPages) setCurrentPage(page);
  };

  const handleSearch = (e) => {
    setSearchTerm(e.target.value);
    setCurrentPage(1);
  };

  const handleSelect = (status) => {
    setSelectedStatus(status);
    setOpen(false);
  };

  // Only render after user is loaded on client
  if (loading || !user) {
    return (
      <Layout>
        <div className="max-w-full mx-auto my-10">
          <h3 className="mb-5 font-semibold text-[#373737] text-2xl">Content Scan Results</h3>
          <div>Loading...</div>
        </div>
      </Layout>
    );
  }

  return (
    <Layout>
      <div className="max-w-full mx-auto my-10">
        <h3 className="mb-5 font-semibold text-[#373737] text-2xl">Content Scan Results</h3>
        <div className="border-b border-gray-200 mb-10">
          <div className="inline-flex relative">
            <Link href="/content-management">
              <button className="text-emerald-500 font-normal px-4 py-2" style={{ cursor: 'pointer' }}>
                All Books
              </button>
            </Link>
            <span className="absolute bottom-0 left-0 h-1 w-full bg-emerald-500 rounded"></span>
          </div>
        </div>

        <div className="flex items-center justify-between pb-2">
          <div>
            <h2 className="text-lg font-semibold text-black">Book Name</h2>
            <p className="text-sm text-gray-600">{book?.name || "Unknown"}</p>
          </div>
          <div className="flex items-center space-x-6">
            <div className="relative">
              <button
                onClick={() => setOpen(!open)}
                className="flex items-center justify-between w-38 border border-[#E8E8E8] rounded px-3 py-1.5 text-sm bg-white hover:bg-gray-50"
              >
                {selectedStatus}
                {open ? (
                  <Image src={uprow} className="ml-2" alt="up-arrow" />
                ) : (
                  <Image src={droparrowdown} className="ml-2" alt="drop-arrow" />
                )}
              </button>
              {open && (
                <div className="absolute right-0 mt-0 w-38 bg-white border border-[#E8E8E8] z-10 rounded text-sm">
                  <button
                    onClick={() => handleSelect('Done')}
                    className="block w-full text-left px-4 py-2 hover:bg-gray-100"
                  >
                    Done
                  </button>
                  <button
                    onClick={() => handleSelect('Not Done')}
                    className="block w-full text-left px-4 py-2 hover:bg-gray-100"
                  >
                    Not Done
                  </button>
                </div>
              )}
            </div>
            {/* Matches Found */}
            <div className="flex">
              <p className="text-sm text-gray-500 border-e pe-4">Matches Found</p>
              <p className="text-xl font-semibold text-gray-800 pl-4">{results.length}</p>
            </div>
          </div>
        </div>

        {/* Search bar */}
        <div className="flex items-center w-full max-w-full space-x-2 mb-4">
          <div className="flex items-center flex-grow border border-gray-300 rounded-md px-3">
            <Image src={search} width={20} alt="search" />
            <input
              type="text"
              placeholder="Search by key, video, or channel"
              className="w-full pl-2 text-sm placeholder-gray-400 focus:outline-none leading-[56px]"
              value={searchTerm}
              onChange={handleSearch}
            />
          </div>
        </div>

        <div className="table_wpr pb-5 mt-5">
          <div className="relative overflow-x-auto">
            <table className="glob_table">
              <thead>
                <tr>
                  <th>Sl. No</th>
                  <th>Search Key</th>
                  <th>YouTube Video</th>
                  <th>Link</th>
                  <th>Channel Name</th>
                  <th>Reviews</th>
                </tr>
              </thead>
              <tbody>
                {loadingA ? (
                  <tr>
                    <td colSpan={6} className="text-center">Loading...</td>
                  </tr>
                ) : filteredResults.length === 0 ? (
                  <tr>
                    <td colSpan={6} className="text-center">No results found.</td>
                  </tr>
                ) : (
                  currentResults.map((result, r) => (
                    <tr key={r}>
                      <td>{indexOfFirstResult + r + 1}</td>
                      <td>{result.keyword}</td>
                      <td className="text-center">{result.video}</td>
                      <td className="text-center">
                        {result.link ? (
                          <a href={result.link} className="text-primary" target="_blank" rel="noopener noreferrer">Watch now</a>
                        ) : (
                          "-"
                        )}
                      </td>
                      <td className="text-center">{result.channel?.name}</td>
                      <td>
                        <div className="flex justify-center">
                          {result.review && result.review.length > 0 ? result.review : 'No reviews yet'}
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>

        {/* Pagination Controls */}
        <div className="flex justify-end mt-8">
          <ul className="flex items-center space-x-2">
            <li
              className={`px-3 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${currentPage === 1 ? 'opacity-50 cursor-not-allowed' : ''}`}
              onClick={() => handlePageChange(currentPage - 1)}
            >
              Prev
            </li>
            {[...Array(totalPages)].map((_, idx) => (
              <li
                key={idx}
                className={`w-8 h-8 border border-[#c9c9c9] flex items-center justify-center ${currentPage === idx + 1 ? 'bg-primary text-white' : 'bg-white text-black'} text-sm font-medium rounded cursor-pointer`}
                onClick={() => handlePageChange(idx + 1)}
              >
                {idx + 1}
              </li>
            ))}
            <li
              className={`px-3 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${currentPage === totalPages ? 'opacity-50 cursor-not-allowed' : ''}`}
              onClick={() => handlePageChange(currentPage + 1)}
            >
              Next
            </li>
          </ul>
        </div>
      </div>
    </Layout>
  );
}