var express = require('express');
var router = express.Router();
const { MongoClient, ObjectId } = require('mongodb');
const crypto = require('crypto'); // Added for password hashing
var dburl = process.env.MONGODB_URL;



/* GET users listing. */
router.get('/list', async function(req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const collection = db.collection("users");

    // Aggregate with lookup to activity_log where userId matches and type is "Login"
    const data = await collection.aggregate([
      {
        $match: { userType: { $in: ["Au", "Pu"] } }
      },
      {
        $lookup: {
          from: "activity_log",
          let: { userId: "$_id" },
          pipeline: [
            { $match: { $expr: { $and: [
              { $eq: ["$userId", "$$userId"] },
              { $eq: ["$type", "Login"] }
            ]}}},
            { $sort: { timestamp: -1 } }
          ],
          as: "activityLog"
        }
      }
    ]).toArray();

    res.json(data.map((item, idx) => ({
      id: idx + 1,
      userID: item._id.toString(),
      name: item.name || 'Unknown',
      email: item.email || 'Unknown',
      plan: item.plan || 'Unknown',
      status: item.status || 'Unknown',
      userType: item.userType || 'Unknown',
      createdAt: item.createdAt || '',
      activityLog: item.activityLog || []
    })));
  } catch (err) {
    console.error('Error fetching users:', err);
    res.status(500).json({ error: 'Internal server error.' });
  } finally {
    await client.close();
  }
});  
router.get('/:userId', async function(req, res, next) {
  const userId = req.params.userId;
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const collection = db.collection("users");

    // Find user by ID
    const user = await collection.findOne({ _id: new ObjectId(userId) });
    if (!user) {
      return res.status(404).json({ error: 'User not found.' });
    }

    
    res.json({
      id: user._id.toString(),
      name: user.name || 'Unknown',
      email: user.email || 'Unknown',
      password: user.password || 'Unknown',
      status: user.status || 'Unknown',
      userType: user.userType || 'Unknown',
      createdAt: user.createdAt || null,
      expiryDate: user.expiryDate || null,
    });
  } catch (err) {
    console.error('Error fetching user:', err);
    res.status(500).json({ error: 'Internal server error.' });
  } finally {
    await client.close();
  }
});

router.put('/:userId', async function(req, res, next) {
  const userId = req.params.userId;
  const updateData = req.body; // Assuming the request body contains the fields to update
  const client = new MongoClient(dburl);
  
  try {
    await client.connect();
    const db = client.db();
    const collection = db.collection("users");

    // Update user by ID
    const result = await collection.updateOne(
      { _id: new ObjectId(userId) },
      { $set: updateData }
    );

    if (result.modifiedCount === 0) {
      return res.status(404).json({ error: 'User not found or no changes made.' });
    }

    res.json({ message: 'User updated successfully.' });
  } catch (err) {
    console.error('Error updating user:', err);
    res.status(500).json({ error: 'Internal server error.' });
  } finally {
    await client.close();
  }
});   

router.put('/change-password/:userId', async function(req, res, next) {
  const userId = req.params.userId;
  const { oldPassword, newPassword } = req.body;
  const client = new MongoClient(dburl);

  // Basic validation
  if (!oldPassword || !newPassword) {
    return res.status(400).json({ error: "Old password and new password are required." });
  }

  if (oldPassword === newPassword) {
    return res.status(400).json({ error: "New password cannot be the same as the old password." });
  }

  // NOTE: In a production environment, you would typically have a JWT authentication middleware
  // that verifies the token and ensures the user making the request is authorized to change
  // the password for the given userId. This implementation assumes such a middleware is
  // either in place or not required for this specific context.

  try {
    await client.connect();
    const db = client.db();
    const collection = db.collection("users");

    // Hash the incoming old password with MD5 (consistent with signup/signin)
    const hashedOldPassword = crypto.createHash('md5').update(oldPassword).digest('hex');

    // Find user by ID and verify old password
    const user = await collection.findOne({ _id: new ObjectId(userId), password: hashedOldPassword });

    if (!user) {
      return res.status(401).json({ error: "Invalid old password or user not found." });
    }

    // Hash the new password with MD5
    const hashedNewPassword = crypto.createHash('md5').update(newPassword).digest('hex');

    // Update user's password
    await collection.updateOne({ _id: new ObjectId(userId) }, { $set: { password: hashedNewPassword, updatedAt: new Date() } });

    res.json({ message: 'Password updated successfully.' });
  } catch (err) {
    console.error('Error changing password:', err);
    res.status(500).json({ error: 'Internal server error.' });
  } finally {
    await client.close();
  }
});




module.exports = router;
