var express = require('express');
var router = express.Router();
const { MongoClient, ObjectId } = require('mongodb');
var dburl = process.env.MONGODB_URL;
// console.log('MONGODB_URL:', dburl);


/* GET home page. */
router.get('/', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const uploadsCollection = db.collection(req.query.tab); // usually "uploads"
    const processQueueCollection = db.collection("process_queue");
    const authID = req.query.authID;
    const keyName = req.query.keyName;
    const keyValue = req.query.keyValue;
    const query = {};
    if (keyName && keyValue) {
      query[keyName] = keyName === '_id' ? new ObjectId(keyValue) : keyValue;
    }
    query['authID'] = authID;

    // Join uploads with process_queue and count process_queue rows for each upload
    const data = await uploadsCollection.aggregate([
      { $match: query },
      {
        $lookup: {
          from: "process_queue",
          localField: "_id",
          foreignField: "itemid",
          as: "processQueueRows"
        }
      },
      {
        $addFields: {
          processQueueCount: { $size: "$processQueueRows" }
        }
      }
    ]).toArray();

    const results = data.map((item, idx) => ({
      itemid: item._id.toString(),
      id: idx + 1,
      name: item.title || 'Unknown',
      author: item.author || 'Unknown',
      publisher: item.publisher || 'Unknown',
      keywords: item.keywords || 'Unknown',
      date: item.uploadDate ? new Date(item.uploadDate).toLocaleDateString() : '',
      status: 'Processing',
      rescan: item.processQueueCount || 0
    }));
    res.json(results);
  } finally {
    await client.close();
  }
}
);
router.get('/adminbook', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const uploadsCollection = db.collection(req.query.tab); // usually "uploads"
    const processQueueCollection = db.collection("process_queue");
    const authID = req.query.authID;
    const keyName = req.query.keyName;
    const keyValue = req.query.keyValue;
    const query = {};
    if (keyName && keyValue) {
      query[keyName] = keyName === '_id' ? new ObjectId(keyValue) : keyValue;
    }
    // query['authID'] = authID;

    // Join uploads with process_queue and count process_queue rows for each upload
    const data = await uploadsCollection.aggregate([
      { $match: query },
      {
        $lookup: {
          from: "process_queue",
          localField: "_id",
          foreignField: "itemid",
          as: "processQueueRows"
        }
      },
      {
        $addFields: {
          processQueueCount: { $size: "$processQueueRows" }
        }
      }
    ]).toArray();

    const results = data.map((item, idx) => ({
      itemid: item._id.toString(),
      id: idx + 1,
      name: item.title || 'Unknown',
      author: item.author || 'Unknown',
      publisher: item.publisher || 'Unknown',
      keywords: item.keywords || 'Unknown',
      date: item.uploadDate ? new Date(item.uploadDate).toLocaleDateString() : '',
      status: 'Processing',
      rescan: item.processQueueCount || 0
    }));
    res.json(results);
  } finally {
    await client.close();
  }
}
);
// Helper function to get user data by authID
async function getUserData(usersCollection, authID) {
  try {
    const user = await usersCollection.findOne({ _id: new ObjectId(authID) });
    return user;
  } catch (e) {
    return null;
  }
}

router.get('/admin', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const uploadsCollection = db.collection("uploads");
    const usersCollection = db.collection("users");

    // Join uploads with process_queue, order by uploadDate desc
    const data = await uploadsCollection.aggregate([
      {
        $lookup: {
          from: "process_queue",
          localField: "_id",
          foreignField: "itemid",
          as: "processQueueRows"
        }
      },
      {
        $addFields: {
          processQueueCount: { $size: "$processQueueRows" }
        }
      },
      {
        $sort: { uploadDate: -1 }
      }
    ]).toArray();

    // Attach user data using getUserName helper
    const results = await Promise.all(data.map(async (item, idx) => {
      const user = item.authID ? await getUserData(usersCollection, item.authID) : null;
      return {
        itemid: item._id.toString(),
        id: idx + 1,
        title: item.title || 'Unknown',
        user: user || {},
        author: item.author || 'Unknown',
        publisher: item.publisher || 'Unknown',
        keywords: item.keywords || 'Unknown',
        uploadDate: item.uploadDate ? new Date(item.uploadDate).toLocaleDateString() : '',
        status: 'Processing',
        rescan: item.processQueueCount || 0
      };
    }));

    res.json(results);
  } finally {
    await client.close();
  }
}
);
router.get('/searchresult', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const process_queue_collection = db.collection('process_queue');
    const itemid = req.query.itemid;

    // Join process_queue with search_result where process_queue._id == search_result.processQueueId
    const data = await process_queue_collection.aggregate([
      { $match: { itemid: new ObjectId(itemid) } },
      {
        $lookup: {
          from: 'search_result',
          localField: '_id',
          foreignField: 'processQueueId',
          as: 'searchResults'
        }
      }
    ]).toArray();
    console.log('Data:', data);

    const results = [];
    data.forEach((item) => {
      console.log(item);
      if (
        item.searchResults &&
        item.searchResults.length > 0 &&
        item.searchResults[0].result &&
        item.searchResults[0].result.video_results
      ) {
        console.log(item.searchResults[0].result.video_results);

        item.searchResults[0].result.video_results.forEach((row, idx) => {
          results.push({
            itemid: item.itemid,
            id: idx + 1,
            keyword: item.searchKey,
            video: row.title || 'Unknown',
            link: row.link || '#',
            channel: row.channel || [],
            occurence: 0,
            rescan: 0,
            confidence: 0,
          });
        });
      }
    });

    res.json(results);
  } finally {
    await client.close();
  }
});
router.get('/searchmatchcount', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const process_queue_collection = db.collection('process_queue');
    const itemid = req.query.itemid;

    // Only match process_queue documents for the given itemid
    const matchStage = itemid
      ? { $match: { itemid: new ObjectId(itemid) } }
      : { $match: {} };

    // Join process_queue with search_result where process_queue._id == search_result.processQueueId
    const data = await process_queue_collection.aggregate([
      matchStage,
      {
        $lookup: {
          from: 'search_result',
          localField: '_id',
          foreignField: 'processQueueId',
          as: 'searchResults'
        }
      }
    ]).toArray();

    let count = 0;
    data.forEach((item) => {
      if (
        item.searchResults &&
        item.searchResults.length > 0 &&
        item.searchResults[0].result &&
        item.searchResults[0].result.video_results
      ) {
        count += item.searchResults[0].result.video_results.length;
      }
    });

    res.json({ count });
  } finally {
    await client.close();
  }
});
router.get('/pendingrescan', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const process_queue_collection = db.collection('process_queue');
    const status = "Queued";

    // Find all documents with status "Queued"
    const count = await process_queue_collection.countDocuments({ status: status });

    res.json({ count });
  } finally {
    await client.close();
  }
});
router.get('/activity-log', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const collection = db.collection('activity_log');

    // Join with users, order by timestamp DESC, limit 5
    const logs = await collection.aggregate([
      {
        $lookup: {
          from: "users",
          localField: "userId",
          foreignField: "_id",
          as: "user"
        }
      },
      { $unwind: { path: "$user", preserveNullAndEmptyArrays: true } },
      { $sort: { timestamp: -1 } },
      { $limit: 5 }
    ]).toArray();

    res.json(logs);
  } finally {
    await client.close();
  }
});

router.get('/plans', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const collection = db.collection('plans');

    // Fetch all plans
    const plans = await collection.find({}).toArray();

    res.json(plans);
  } finally {
    await client.close();
  }
});

router.post('/plan-update', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const collection = db.collection('plans');
    
    const { planId, name, desc, status, price, validity, searchCount } = req.body;

    // Construct the updates object
    const updates = {};
    if (name) updates.name = name;
    if (desc) updates.desc = desc;
    if (status !== undefined) updates.status = status; // Allow boolean values
    if (price) updates.price = price;
    if (validity) updates.validity = validity;
    if (searchCount) updates.searchCount = searchCount;

    // Update the plan with the given planId
    const result = await collection.updateOne(
      { _id: new ObjectId(planId) },
      { $set: updates }
    );

    if (result.modifiedCount > 0) {
      res.json({ success: true, message: 'Plan updated successfully' });
    } else {
      res.json({ success: false, message: 'No changes made to the plan' });
    }
  } catch (err) {
    console.error('Error updating plan:', err);
    res.status(500).json({ success: false, message: 'Internal server error' });
  } finally {
    await client.close();
  }
});

 

router.get('/search-by-days', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();
    const processQueueCollection = db.collection("search_result");

    const today = new Date();
    today.setHours(23, 59, 59, 999); // End of today
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(today.getDate() - 39); // Last 30 days including today
    thirtyDaysAgo.setHours(0, 0, 0, 0);

    // Generate a list of the last 30 days for filling in missing dates
    const dateRange = [];
    for (let i = 0; i < 30; i++) {
      const d = new Date(thirtyDaysAgo);
      d.setDate(thirtyDaysAgo.getDate() + i);
      // Format as 'DD/MM'
      const day = d.toLocaleString('en-GB', { day: '2-digit' });
      const month = d.toLocaleString('en-GB', { month: '2-digit' });
      dateRange.push(`${day}/${month}`);
    }

    // Aggregate counts by day and month
    const aggregationPipeline = [
      {
        $match: {
          createdAt: {
            $gte: thirtyDaysAgo,
            $lte: today
          }
        }
      },
      {
        $group: {
          _id: {
            day: { $dayOfMonth: "$createdAt" },
            month: { $month: "$createdAt" }
          },
          value: { $sum: 1 }
        }
      },
      {
        $project: {
          _id: 0,
          date: {
            $concat: [
              { $toString: { $cond: [ { $lt: ["$_id.day", 10] }, { $concat: ["0", { $toString: "$_id.day" }] }, { $toString: "$_id.day" } ] } },
              "/",
              { $toString: { $cond: [ { $lt: ["$_id.month", 10] }, { $concat: ["0", { $toString: "$_id.month" }] }, { $toString: "$_id.month" } ] } }
            ]
          },
          value: 1
        }
      },
      {
        $sort: { date: 1 }
      }
    ];

    const dailyCounts = await processQueueCollection.aggregate(aggregationPipeline).toArray();

    // Create a map for quick lookup of counts by date
    const dailyCountsMap = new Map();
    dailyCounts.forEach(item => dailyCountsMap.set(item.date, item.value));

    // Combine the full date range with the actual counts
    const finalResults = dateRange.map(date => ({
      date: date,
      value: dailyCountsMap.get(date) || 0
    }));

    res.json(finalResults);

  } catch (err) {
    console.error('Error fetching search by days data:', err);
    res.status(500).json({ success: false, message: 'Internal server error' });
  } finally {
    await client.close();
  }
});


router.get('/top-active-users', async function (req, res, next) {
  const client = new MongoClient(dburl);
  try {
    await client.connect();
    const db = client.db();

    // Aggregate search_result → process_queue → uploads, group by uploads.authID
    const pipeline = [
      // Join search_result with process_queue
      {
        $lookup: {
          from: "process_queue",
          localField: "processQueueId",
          foreignField: "_id",
          as: "processQueueInfo"
        }
      },
      { $unwind: "$processQueueInfo" },
      // Join process_queue with uploads
      {
        $lookup: {
          from: "uploads",
          localField: "processQueueInfo.itemid",
          foreignField: "_id",
          as: "uploadInfo"
        }
      },
      { $unwind: "$uploadInfo" },
      // Group by uploads.authID (user)
      {
        $group: {
          _id: "$uploadInfo.authID",
          searchResultCount: { $sum: 1 }
        }
      },
      // Lookup user name from users collection
      {
        $lookup: {
          from: "users",
          let: { userId: "$_id" },
          pipeline: [
            {
              $match: {
                $expr: {
                  $eq: [
                    "$_id",
                    { $toObjectId: "$$userId" }
                  ]
                }
              }
            }
          ],
          as: "userInfo"
        }
      },
      { $unwind: { path: "$userInfo", preserveNullAndEmptyArrays: true } },
      {
        $lookup: {
          from: "activity_log",
          let: { userId: "$_id" },
          pipeline: [
            {
              $match: {
                $expr: {
                  $eq: [
                    "$userId",
                    { $toObjectId: "$$userId" }
                  ]
                }
              }
            },
            { $sort: { timestamp: -1 } },
            { $limit: 1 }
          ],
          as: "lastActivity"
        }
      },
      {
        $project: {
          userId: "$_id",
          name: { $ifNull: ["$userInfo.name", "Unknown"] },
          searchResultCount: 1,
          lastActivity: {
            $ifNull: [{ $arrayElemAt: ["$lastActivity.timestamp", 0] }, null]
          }
        }
      },
      {
        $sort: {
          searchResultCount: -1,
          lastActivity: -1
        }
      }
    ];

    const topActiveUsers = await db.collection("search_result").aggregate(pipeline).toArray();
    res.json(topActiveUsers);

  } catch (err) {
    console.error('Error fetching top active users:', err);
    res.status(500).json({ error: 'Internal server error.' });
  } finally {
    await client.close();
  }
});



module.exports = router;
