var express = require('express');
var router = express.Router();
const { MongoClient, ObjectId } = require('mongodb');
var dburl = process.env.MONGODB_URL;

router.get('/', async function (req, res, next) {
    const client = new MongoClient(dburl);
    try {
        await client.connect();
        const db = client.db();
        const collection = db.collection("coupons");

        // Fetch all coupons
        const data = await collection.find({}).toArray();

        res.json(data.map((item, idx) => ({
            id: idx + 1,
            couponID: item._id.toString(),
            code: item.code || 'Unknown',
            discount: item.discount || 'Unknown',
            discountType: item.discountType || 'Unknown',
            description: item.description || 'No description',
            validFromDate: item.validFromDate ? item.validFromDate : 'Unknown',
            validToDate: item.validToDate || 'Unknown',
            status: item.status || 'Unknown',
            createdAt: item.createdAt || ''
        })));
    } catch (err) {
        console.error('Error fetching coupons:', err);
        res.status(500).json({ error: 'Internal server error.' });
    } finally {
        await client.close();
    }
});
router.post('/create', async function (req, res, next) {
    const client = new MongoClient(dburl);
    try {
        await client.connect();
        const db = client.db();
        const collection = db.collection("coupons");
        console.log('Creating coupon with data:', req.body);
        
        // Create a new coupon
        const newCoupon = {
            code: req.body.code,
            discount: req.body.discount,
            discountType: req.body.discount_type || 'Percentage', // Default to 'Percentage' if not provided
            validFromDate: new Date(req.body.valid_from),
            validToDate: new Date(req.body.valid_to),
            description: req.body.description || '',
            status: req.body.status || 'Active',
            createdAt: new Date()
        };

        const result = await collection.insertOne(newCoupon);
        res.status(201).json({ message: 'Coupon created successfully', couponId: result.insertedId });
    } catch (err) {
        console.error('Error creating coupon:', err);
        res.status(500).json({ error: 'Internal server error.' });
    } finally {
        await client.close();
    }
});
router.put('/update/:id', async function (req, res, next) {
    const client = new MongoClient(dburl);
    try {
        await client.connect();
        const db = client.db();
        const collection = db.collection("coupons");

        // Update a coupon by ID
        const couponId = req.params.id;
        console.log('Updating coupon with ID:', couponId, 'and data:', req.body);
        const validfromDate = new Date(req.body.valid_from);
        const validtoDate = new Date(req.body.valid_to);

        // console.log('Valid From Date:', validfromDate);
        // console.log('Valid To Date:', validtoDate);
        const updateData = {
            code: req.body.code,
            discount: req.body.discount,
            discountType: req.body.discount_type || 'Percentage', // Default to 'Percentage' if not provided
            description: req.body.description || '',
            validFromDate:  validfromDate,
            validToDate: validtoDate,
            status: req.body.status || 'Active',
            updatedAt: new Date()
        };

        const result = await collection.updateOne({ _id: new ObjectId(couponId) }, { $set: updateData });
        if (result.modifiedCount > 0) {
            res.json({ message: 'Coupon updated successfully' });
        } else {
            res.status(404).json({ error: 'Coupon not found' });
        }
    } catch (err) {
        console.error('Error updating coupon:', err);
        res.status(500).json({ error: 'Internal server error.' });
    } finally {
        await client.close();
    }
});
router.delete('/delete/:id', async function (req, res, next) {
    const client = new MongoClient(dburl);
    try {
        await client.connect();
        const db = client.db();
        const collection = db.collection("coupons");

        // Delete a coupon by ID
        const couponId = req.params.id;
        const result = await collection.deleteOne({ _id: new ObjectId(couponId) });
        if (result.deletedCount > 0) {
            res.json({ message: 'Coupon deleted successfully' });
        } else {
            res.status(404).json({ error: 'Coupon not found' });
        }
    } catch (err) {
        console.error('Error deleting coupon:', err);
        res.status(500).json({ error: 'Internal server error.' });
    } finally {
        await client.close();
    }
});

router.get('/apply/:id', async function (req, res, next) {
    console.log('Applying coupon with ID:', req.params.id);
    
    const client = new MongoClient(dburl);
    try {
        await client.connect();
        const db = client.db();
        const collection = db.collection("coupons");

        // Apply a coupon by ID
        const couponId = req.params.id;
        console.log('Applying coupon with ID:', couponId);
        const coupon = await collection.findOne({ code: couponId });
        if (coupon) {
            res.json({
                message: 'Coupon applied successfully',
                coupon: {
                    code: coupon.code,
                    discount: coupon.discount,
                    discountType: coupon.discountType,
                    description: coupon.description,
                    validFromDate: coupon.validFromDate,
                    validToDate: coupon.validToDate,
                    status: coupon.status
                }
            });
        } else {
            res.status(404).json({ error: 'Coupon not found' });
        }
    } catch (err) {
        console.error('Error applying coupon:', err);
        res.status(500).json({ error: 'Internal server error.' });
    } finally {
        await client.close();
    }
});
module.exports = router;