const express = require('express');
const router = express.Router();
const { MongoClient, ObjectId } = require('mongodb');
const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY); // Initialize Stripe with your secret key
var dburl = process.env.MONGODB_URL;

// Create a new order with a plan and optional coupon
router.post('/create', async (req, res) => {
  const { planId, planName, planPrice, couponcode } = req.body;
  console.log('Creating order with data:', req.body);
  console.log('user:', req.user);

  const client = new MongoClient(dburl); // Ensure client is initialized properly

  try {
    await client.connect();
    const db = client.db();
    const ordersCollection = db.collection("orders");
    const couponsCollection = db.collection("coupons");

    // Validate couponcode if provided
    let coupon = null;
    if (couponcode) {
      coupon = await couponsCollection.findOne({ code: couponcode }); // Use `findOne` instead of `find` for a single document
      console.log('Coupon found:', coupon);

      if (!coupon) {
        return res.status(404).json({ error: 'Coupon not found.' });
      }
    }

    // Create a new order
    const newOrder = {
      userID: req.user?.userId, // Ensure `req.user` is set by middleware
      planId,
      planName,
      planPrice,
      coupon: couponcode ? couponcode : null, // Associate coupon if provided
      createdAt: new Date(),
      status: 'Pending', // Default status
      stripe_session_ID: null,
      stripe_payment_intent: null,
      payment_time: null,
    };

    const result = await ordersCollection.insertOne(newOrder);
    res.status(201).json({ message: 'Order created successfully', order: newOrder, url: `/checkout/${result.insertedId}` });
  } catch (err) {
    console.error('Error creating order:', err);
    res.status(500).json({ error: 'Failed to create order.' });
  } finally {
    await client.close(); // Ensure the database connection is closed
  }
});

router.get('/:orderId', async (req, res) => {
  const orderId = req.params.orderId;
  console.log('Fetching order with ID:', orderId);

  const client = new MongoClient(dburl); // Ensure client is initialized properly

  try {
    await client.connect();
    const db = client.db();
    const ordersCollection = db.collection("orders");

    // Fetch the order by ID
    const order = await ordersCollection.findOne({ _id: new ObjectId(orderId) });
    if (!order) {
      return res.status(404).json({ error: 'Order not found.' });
    }

    res.json(order);
  } catch (err) {
    console.error('Error fetching order:', err);
    res.status(500).json({ error: 'Failed to fetch order.' });
  } finally {
    await client.close(); // Ensure the database connection is closed
  }
});

// Create a checkout session with Stripe
router.post('/create-checkout-session', async (req, res) => {
    const id = req.body.id;
    const client = new MongoClient(dburl);
    await client.connect();
    const db = client.db();
    const ordersCollection = db.collection("orders");

    // Fetch the order by ID
    const order = await ordersCollection.findOne({ _id: new ObjectId(id) });
    if (!order) {
      return res.status(404).json({ error: 'Order not found.' });
    }
    console.log('Creating Stripe Checkout session for order:', order);

    const unitPrice = parseInt(order.planPrice * 100); // Convert to cents for Stripe
    console.log('Unit price in cents:', unitPrice);
    
  try {
    // Create a Stripe Checkout session
    const session = await stripe.checkout.sessions.create({
      payment_method_types: ['card'], // Accept card payments
      line_items: [
        {
          price_data: {
            currency: 'usd', // Currency (e.g., 'usd')
            product_data: {
              name: order.planName, // Name of the plan
            },
            unit_amount: unitPrice, // Stripe expects the amount in cents
          },
          quantity: 1, // Quantity of the item
        },
      ],
      mode: 'payment', // Payment mode
      success_url: `${process.env.FRONTEND_URL}/success?session_id={CHECKOUT_SESSION_ID}&oid=${id}`, // Redirect URL on success
      cancel_url: `${process.env.FRONTEND_URL}/cancel?oid=${id}`, // Redirect URL on cancellation
    });

    // Respond with the session ID
    res.status(200).json({ sessionId: session.id });
  } catch (err) {
    console.error('Error creating Stripe Checkout session:', err);
    res.status(500).json({ error: 'Failed to create Stripe Checkout session.' });
  }
});


router.get('/stripe-session/:sessionId/:oid', async (req, res) => {
  const { sessionId, oid } = req.params;
  const client = new MongoClient(dburl);

  try {
    // Retrieve Stripe session
    const session = await stripe.checkout.sessions.retrieve(sessionId);

    await client.connect();
    const db = client.db();
    const ordersCollection = db.collection("orders");
    const planCollection = db.collection("plans");
    const usersCollection = db.collection("users");

    // Fetch the order by ID
    const order = await ordersCollection.findOne({ _id: new ObjectId(oid) });
    if (!order) {
      return res.status(404).json({ error: 'Order not found.' });
    }
     


    // Update order fields
    const updateFields = {
      status: 'Paid',
      stripe_session_ID: session.id,
      stripe_payment_intent: session.payment_intent,
      payment_time: new Date(),
    };

    await ordersCollection.updateOne(
      { _id: new ObjectId(oid) },
      { $set: updateFields }
    );

    if (order) {
      const planDetails = await planCollection.findOne({ _id: new ObjectId(order.planId) });
      if (planDetails) {
        const userUpdateFields = {
          expiryDate: new Date(Date.now() + parseInt(planDetails.validity) * 24 * 60 * 60 * 1000), // Set expiry date to validity days from now
          searchCount: planDetails.searchCount, // Set search count from plan
          activeOrder: order._id, // Set active order from order

        };
        await usersCollection.updateOne(
          { _id: new ObjectId(order.userID) },
          { $set: userUpdateFields }
        );
      }
    }

    // Fetch the updated order
    const updatedOrder = await ordersCollection.findOne({ _id: new ObjectId(oid) });

   


    res.status(200).json({ order: updatedOrder, session });
  } catch (err) {
    console.error('Error processing Stripe session/order:', err);
    res.status(500).json({ error: 'Failed to process Stripe session/order.' });
  } finally {
    await client.close();
  }
});

router.get('/cancel/:oid', async (req, res) => {
  const { oid } = req.params;
  const client = new MongoClient(dburl);

  try {

    await client.connect();
    const db = client.db();
    const ordersCollection = db.collection("orders");

    // Fetch the order by ID
    const order = await ordersCollection.findOne({ _id: new ObjectId(oid) });
    if (!order) {
      return res.status(404).json({ error: 'Order not found.' });
    }

    // Update order fields
    const updateFields = {
      status: 'Canceled',
    
    };

    await ordersCollection.updateOne(
      { _id: new ObjectId(oid) },
      { $set: updateFields }
    );

    // Fetch the updated order
    const updatedOrder = await ordersCollection.findOne({ _id: new ObjectId(oid) });

    res.status(200).json({ order: updatedOrder});
  } catch (err) {
    console.error('Error processing order:', err);
    res.status(500).json({ error: 'Failed to process order.' });
  } finally {
    await client.close();
  }
});

module.exports = router;
